/* main.c - Hello World example for S32R274 */
/* Description:  Configures CLKOUT and blinks LED */
/* Rev 1.0 Aug 2 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main_Z4.c
* Owner             David Chung
* Version           1.0
* Date              Aug-2-2018
* Classification    General Business Information
* Brief             Produces S32R274 CLKOUT
********************************************************************************
* Detailed Description:
* S32R274's default clock source is 16 MHz IRCOSC.  This code example
* divides IRCOSC by 10 for output frequency of 1.6 MHz.
* Configures S32R274 CLKOUT signal to 1.6 MHz. GPIO also
* flashes based on internal counter to show that code example
* is running. Connect the GPIO to a user LED to observe flashing.
*
* Refer to "CLKOUT Waveform.png" for expected oscilloscope output
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB + MPC57XXMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            16 MHz IRC
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Connect pin P9.7 (CLK_OUT0) to an oscilloscope to check
* 					CLK_OUT0 frequency. The CLKOUT pin displays the IRC
* 					frequency divided by 10 (1.6 MHz).
* 					Connect P8.1 (PA0) to any user LED P7.x	to indicate app is running.
*
* 					Refer to "Hardware Setup.jpg"
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-02-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

extern void xcptn_xmpl(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

int main(void)
{
	int counter = 0;

	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	/* S32R27 boots from the 16 MHz IRCOSC.
	 * MCB controls CLKOUT source selects.
	 */
	MCB.CLKOUT_SEL.B.CLKOUT0 = 0b0000; //Select IRC_CLK for CLKOUT0
	CGM.AC14_DC0.R = 0x80090000; //Divide 16 MHz IRC_CLK by (9+1) = 10 for output of 1.6 MHz

	/* CLKOUT0 maps to PB6 (MSCR22) */
	SIUL2.MSCR[22].B.SSS = 1; //Configure pin for CLKOUT0
	SIUL2.MSCR[22].B.OBE = 1; //Enable output buffer

	/* Configure GPIO PA0 (P8.1) */
	SIUL2.MSCR[0].B.OBE = 1;
	SIUL2.GPDO[0].B.PDO_4n = 1;

	/* Loop forever */
	for(;;) {	   
	   	counter++;

	   	/* Wait 50000 counts of internal counter
	   	 * before toggling GPIO.
	   	 */
	   	if(counter >= 50000){
	   		SIUL2.GPDO[0].B.PDO_4n ^= 1; //Toggle PA0
	   		counter = 0; //Reset counter
	   	}
	}
}


